#!/bin/sh
#
# NAG Fortran Compiler unattended installation script.
#
# Target installation directories must be specified via environment variables:
#
#   INSTALL_TO_BINDIR
#     the directory to contain binary materials;
#   INSTALL_TO_LIBDIR
#     the directory to contain supporting library materials, which must
#     be different from the binaries directory;
#   (optional) INSTALL_TO_CATMANDIR
#     the directory prefix for preformatted man pages;
#   (optional) INSTALL_TO_MANDIR
#     the directory prefix for unformatted man pages;
#
# An existing installation found under ${INSTALL_TO_LIBDIR} will be removed.
#
# If running as the root user, the installed files will have their ownership
# set to root and their group ID set to bin.
#
# Release 6.1: INSTALLU.sh 5304 2016-07-22 01:04:12Z Malcolm Cohen
#
# Release 6.2: INSTALLU.sh 6442 2018-07-15 06:06:06Z Malcolm Cohen
#
# Release 7.0: $Id: INSTALLU.sh 8120 2020-12-18 02:21:45Z Malcolm Cohen $
#
if [ -z "${INSTALL_TO_BINDIR-}" ]; then
    echo "Error: Please set INSTALL_TO_BINDIR in the environment."
    exit 1
fi
if [ -z "${INSTALL_TO_LIBDIR-}" ]; then
    echo "Error: Please set INSTALL_TO_LIBDIR in the environment."
    exit 1
fi
remove=rm
removedir=rmdir
removetree="${remove} -rf"
whoami=whoami
defaultlibdir=/usr/local/lib/NAG_Fortran
chown='chown -fhR'
chgrp='chgrp -fhR'
if [ -x /usr/bin/uname -o -x /bin/uname ]; then
    if [ "`uname`" = "SunOS" ]; then
        if expr "`uname -r`" \>= 5.0 >/dev/null; then
            whoami=/usr/ucb/whoami
	    defaultlibdir=/opt/NAG_Fortran/lib
        fi
    elif [ "`uname`" = "HP-UX" -o "`uname`" = "Linux" ]; then
        chown='chown -R'
        chgrp='chgrp -R'
    fi
fi
whoami="`${whoami}`"
move='cp -p'
installer='INSTALLU.sh'
echo
echo "Welcome to the NAG Fortran Compiler unattended installation script."
echo
if [ ! -f ${installer} ]; then
    echo "Error: Not within a NAG Fortran Compiler distribution directory."
    echo
    echo "Please change your working directory to the relevant"
    echo "NAG_Fortran-* distribution directory and try again."
    exit 1
fi
licence_agreement_file=nag_tsandcs.txt
if [ ! -f ${licence_agreement_file} ]; then
    echo "Error: Cannot find licence agreement file ${licence_agreement_file}"
    exit 1
fi
echo "By using this script you agree to the licence terms in the accompanying"
echo "${licence_agreement_file} file."
echo
bindir="${INSTALL_TO_BINDIR}"
if [ -f "${bindir}" ]; then
    echo "Error: ${bindir} is a normal file, not a directory."
    exit 1
fi
case "${bindir}" in
    /* ) pathchk -- "$bindir"
         if [ $? -ne 0 ]; then
             echo "Error: Binary directory is not acceptable."
             exit 1
         fi
       ;;
    * ) echo "Error: Binary directory must be absolute path."
       exit 1
       ;;
esac
echo "Binary materials will be installed in directory ${bindir}"
libdir="${INSTALL_TO_LIBDIR}"
if [ -f "${libdir}" ]; then
    echo "Error: ${libdir} is a normal file, not a directory."
    exit 1
elif [ "${bindir}" = "${libdir}" ]; then
    echo "Error: Library directory must be separate from binary directory."
    exit 1
fi
case "${libdir}" in
    /* ) pathchk -- "$libdir"
         if [ $? -ne 0 ]; then
             echo "Error: Library directory is not acceptable."
             exit 1
         fi
       ;;
    * ) echo "Error: Library directory must be absolute path."
       exit 1
       ;;
esac
echo "Library materials will be installed in directory ${libdir}"
if [ -z "${INSTALL_TO_CATMANDIR-}" ]; then
    echo "No installation of preformatted man pages will be performed."
elif [ ! -f man/nagfor.z ]; then
    echo "No preformatted man pages to install."
else
    catmandir="${INSTALL_TO_CATMANDIR}"
    echo "Preformatted man pages will be installed in directory with prefix" \
         "${catmandir}"
fi
if [ -z "${INSTALL_TO_MANDIR-}" ]; then
    echo "No installation of unformatted man pages will be performed."
else
    mandir="${INSTALL_TO_MANDIR}"
    echo "Unformatted man pages will be installed in directory with prefix" \
         "${mandir}"
fi
if [ -x "${libdir}/forcomp" ]; then
    if [ -x "${libdir}/UNINSTALL_NAGFOR" ]; then
        "${libdir}/UNINSTALL_NAGFOR"
        echo "Previous installation located around ${libdir} has been removed" \
             "using the UNINSTALL_NAGFOR script."
    else
        ${removetree} "${libdir}"
        echo "Previous installation located around ${libdir} has been removed" \
             "using ${removetree}"
    fi
fi
echo
if [ "${whoami}" = "root" ]; then
    owner=root
    group=bin
fi
# Create directories, which has to be done after an old library tree
# has been removed.
for dirtype in bin lib catman1 catman3 man1 man3; do
    if [ "${dirtype}" = "bin" ]; then
        thisdir="${bindir}"
    elif [ "${dirtype}" = "lib" ]; then
        thisdir="${libdir}"
    elif [ "${dirtype}" = "catman1" ]; then
        if [ -z "${catmandir-}" ]; then
            continue
        fi
        thisdir="${catmandir}1"
    elif [ "${dirtype}" = "catman3" ]; then
        if [ -z "${catmandir-}" ]; then
            continue
        fi
        thisdir="${catmandir}3"
    elif [ "${dirtype}" = "man1" ]; then
        if [ -z "${mandir-}" ]; then
            continue
        fi
        thisdir="${mandir}1"
    elif [ "${dirtype}" = "man3" ]; then
        if [ -z "${mandir-}" ]; then
            continue
        fi
        thisdir="${mandir}3"
    fi
    while [ ! -d "${thisdir}" ]; do
        create="${thisdir}"
        parent=`dirname "${create}"`
        while [ ! -d "${parent}" -a -n "${parent}" ]; do
            create="${parent}"
            parent=`dirname "${create}"`
        done
        if mkdir "${create}"; then
            echo "Created ${create}"
            if [ -z "${newdirs-}" ]; then
                newdirs="\"${create}\""
            else
                newdirs="\"${create}\" ${newdirs}"
            fi
        else
            echo "Error: Cannot create ${create}"
            exit 1
        fi
        chmod 755 $create
    done
done
echo
uninstall="${libdir}/UNINSTALL_NAGFOR"
binaries="nagfor nagfmcheck"
if [ -x bin/dbx90 ]; then
    binaries="${binaries} dbx90"
fi
for binary in ${binaries}; do
    $remove -f "${bindir}/${binary}"
    if [ "${binary}" = "nagfor" -a "${defaultlibdir}" = "${libdir}" ] || \
           [ "${binary}" = "dbx90" -a "${defaultlibdir}" = "${libdir}" ] || \
           [ "${binary}" = "nagfmcheck" ]; then
        $move "bin/${binary}" "${bindir}"
        chmod 755 "${bindir}/${binary}"
    else
        $move "bin/${binary}" "${libdir}"
        echo "#!/bin/sh" > "${bindir}/${binary}"
        if [ "${binary}" = "nagfor" ]; then
            echo "\"${libdir}/${binary}\"" '"$@"' -Qpath "\"${libdir}\"" \
                 >> "${bindir}/${binary}"
        elif [ "${binary}" = "dbx90" ]; then
            echo "\"${libdir}/${binary}\"" -Qpath "\"${libdir}\"" '"$@"' \
                 >> "${bindir}/${binary}"
        fi
        chmod 755 "${bindir}/${binary}"
    fi
    if [ "${whoami}" = "root" ]; then
        $chown $owner "${bindir}/${binary}"
        $chgrp $group "${bindir}/${binary}"
    fi
done
echo "Installed bin directory ${bindir}"
echo
for docdir in doc html; do
    if [ -d "${docdir}" ]; then
        tar cf - "${docdir}" | (cd "${libdir}" ; tar xf -)
    else
        echo "??Warning: Missing ${docdir} directory."
    fi
done
cd lib
tar cf - . | (cd "${libdir}" ; tar xf -)
cd ..
if [ "${whoami}" = "root" ]; then
    $chown $owner "${libdir}"
    $chgrp $group "${libdir}"
fi
echo "Installed lib directory ${libdir}"
echo
if [ ! -z "${catmandir-}" ]; then
    for i in nagfor dbx90 fpp nagfmcheck; do
        $move "man/$i.z" "${catmandir}1/$i.Z"
        chmod 644 "${catmandir}1/$i.Z"
        if [ "${whoami}" = "root" ]; then
            $chown $owner "${catmandir}1/$i.Z"
            $chgrp $group "${catmandir}1/$i.Z"
        fi
    done
    for i in f90_gc f90_unix f90_unix_file ieee_features \
                    f90_iostat f90_unix_dir f90_unix_io nag_modules \
                    f90_kind f90_unix_dirent f90_unix_proc \
                    f90_preconn_io f90_unix_env ieee_arithmetic \
                    f90_stat f90_unix_errno ieee_exceptions iso_fortran_env; do
        $move "man/$i.z" "${catmandir}3/$i.Z"
        chmod 644 "${catmandir}3/$i.Z"
        if [ "${whoami}" = "root" ]; then
            $chown $owner "${catmandir}3/$i.Z"
            $chgrp $group "${catmandir}3/$i.Z"
        fi
    done
    echo "Installed preformatted man pages."
    echo
fi
if [ ! -z "${mandir-}" ]; then
    for manext in 1 3; do
        for i in man/*.${manext}; do
            i_base=`basename "$i"`
            $move "$i" "${mandir}${manext}"
            chmod 644 "${mandir}${manext}"
            if [ "${whoami}" = "root" ]; then
                $chown $owner "${mandir}${manext}/${i_base}"
                $chgrp $group "${mandir}${manext}/${i_base}"
            fi
        done
    done
    echo "Installed unformatted man pages."
    echo
fi
tmp_uninstall=`mktemp -t np_INSTXXX`
sed -e "s%NAGFOR_INSTALL_LIBDIR%${libdir}%" \
    -e "s%NAGFOR_INSTALL_BINDIR%${bindir}%" \
    -e "s%REMOVE_DIR%${removedir}%" < UNINSTALL_NAGFOR_core > "${tmp_uninstall}"1
if [ ! -z "${mandir-}" ]; then
    sed -e "s%NAGFOR_INSTALL_MAN1DIR%${mandir}1%" \
        -e "s%NAGFOR_MAN1%%g" \
        -e "s%NAGFOR_INSTALL_MAN3DIR%${mandir}3%" \
        -e "s%NAGFOR_MAN3%%g" < "${tmp_uninstall}"1 > "${tmp_uninstall}"2
else
    sed -e '/NAGFOR_INSTALL_MAN[13]DIR/d' \
        -e '/NAGFOR_MAN[13]/d' < "${tmp_uninstall}"1 > "${tmp_uninstall}"2
fi
if [ ! -z "${newdirs-}" ]; then
    sed -e "s%NAGFOR_NEW_INSTALL_DIRS%${newdirs}%" \
        -e "s%NAGFOR_NEWDIRS%%g" < "${tmp_uninstall}"2 > "${tmp_uninstall}"1
else
    sed -e '/NAGFOR_NEWDIRS/d' < "${tmp_uninstall}"2 > "${tmp_uninstall}"1
fi
if [ "${defaultlibdir}" = "${libdir}" ]; then
    sed -e '/NAGFOR_NONDEFAULT_INSTALL_LIBDIR/d' < "${tmp_uninstall}"1 > "${tmp_uninstall}"2
else
    sed -e "s%NAGFOR_NONDEFAULT_INSTALL_LIBDIR%%g" < "${tmp_uninstall}"1 > "${tmp_uninstall}"2
fi
chmod 755 $bindir/dbx90 $bindir/nagfor $bindir/nagfmcheck >/dev/null 2>&1
find $libdir -type d -exec chmod 755 {} \;
find $libdir -type f -exec chmod 644 {} \;
chmod 755 $libdir/*.exe $libdir/dbx90 $libdir/cforcomp $libdir/forcomp32 $libdir/forcomp $libdir/fpp $libdir/kdongle $libdir/khostid $libdir/klcheck $libdir/kldctl $libdir/nagfor >/dev/null 2>&1
sed '2,3d' < "${tmp_uninstall}"2 > "${uninstall}_core"
sed -e "s%NAGFOR_INSTALL_LIBDIR%${libdir}%" < UNINSTALL_NAGFOR > "${tmp_uninstall}"2
sed '2,3d' < "${tmp_uninstall}"2 > "${uninstall}"
${remove} "${tmp_uninstall}"1 "${tmp_uninstall}"2
chmod 755 "${uninstall}_core" "${uninstall}"
if [ "${whoami}" = "root" ]; then
    $chown $owner "${uninstall}_core" "${uninstall}"
    $chgrp $group "${uninstall}_core" "${uninstall}"
fi
echo "Created uninstall script ${uninstall}"
echo
echo "Installation complete."
echo
echo "For the complete Compiler manual, please see"
echo "  ${libdir}/doc/compiler.pdf"
